"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const runtime_core_1 = require("@postman/runtime.core");
const runtime_mqtt_utils_1 = require("@postman/runtime.mqtt-utils");
const strip_json_comments_1 = __importDefault(require("strip-json-comments"));
exports.default = (async function handler(item, agent, context) {
    const settings = item.payload.settings || {};
    const { variables } = context;
    const { isTLS } = (0, runtime_mqtt_utils_1.parseURL)(item.payload.url);
    const { auth } = item.extensions;
    const tlsOptions = isTLS ?
        {
            rejectUnauthorized: Boolean(settings.strictSSL),
            secureContext: typeof context.secureContext === 'function' ?
                await context.secureContext(item.payload.url)
                : context.secureContext,
        }
        : null;
    const mqttOptions = {
        clientId: item.payload.clientId,
        version: item.payload.version,
        cleanSession: settings.cleanSession,
        keepAlive: settings.keepAlive,
        autoReconnect: settings.autoReconnect,
        properties: item.payload.properties,
    };
    // TODO: Handle with the auth extension
    if (auth && auth.type === 'basic' && auth.basic) {
        auth.basic.forEach((auth) => {
            if (auth.key === 'username' &&
                auth.value !== undefined &&
                typeof auth.value === 'string') {
                mqttOptions.username = auth.value;
            }
            else if (auth.key === 'password' &&
                auth.value !== undefined &&
                typeof auth.value === 'string') {
                mqttOptions.password = auth.value;
            }
        });
    }
    if (item.payload.lastWill?.topic != undefined &&
        item.payload.lastWill?.topic != '') {
        mqttOptions.lastWill = item.payload.lastWill;
    }
    const connection = await agent.connect({
        url: item.payload.url,
        tlsOptions: tlsOptions,
        mqttOptions,
    });
    const events = runtime_core_1.EventChannel.specific(context.events);
    const subscriptionMap = new Map();
    const unsubscriptionMap = new Map();
    // MQTT 5 brokers can send a disconnect packet to the client.
    // This flag tracks if the disconnect event is handled by the client or needs to be emitted on close.
    let receivedDisconnect = false;
    const onSubscribe = (event) => {
        const { name: topic, subscribe, ...options } = event.payload;
        let resolvedTopic = topic;
        let resolvedOptions = options;
        if (variables) {
            resolvedTopic = variables.replaceIn(topic);
            resolvedOptions = variables.replaceIn(options);
        }
        connection.subscribe(resolvedTopic, resolvedOptions);
    };
    const onUnsubscribe = (event) => {
        const topic = event.payload.topic;
        let resolvedTopic = topic;
        if (variables) {
            resolvedTopic = variables.replaceIn(topic);
        }
        try {
            connection.unsubscribe(resolvedTopic);
        }
        catch (err) {
            let message = '';
            if (typeof err === 'string') {
                message = err;
            }
            else if (err instanceof Error) {
                message = err.message;
            }
            events.emit('error', {
                error: `Unable to unsubscribe from topic - ${topic}: ${message}`,
            });
        }
    };
    const onPublish = (event) => {
        if (event.payload.type === 'json') {
            const json = `${event.payload.payload}`;
            event.payload.payload = (0, strip_json_comments_1.default)(json);
        }
        const { topic, payload, type, ...options } = event.payload;
        let resolvedPayload = payload;
        let resolvedTopic = topic;
        let resolvedOptions = options;
        if (variables) {
            resolvedPayload = variables.replaceIn(payload);
            resolvedTopic = variables.replaceIn(topic);
            resolvedOptions = variables.replaceIn(options);
        }
        connection.publish(resolvedTopic, resolvedPayload, resolvedOptions);
    };
    const onDisconnect = () => {
        connection.disconnect();
    };
    return new Promise((resolve) => {
        const onDone = () => {
            connection.disconnect();
            events
                .off('subscribe', onSubscribe)
                .off('unsubscribe', onUnsubscribe)
                .off('publish', onPublish)
                .off('disconnect', onDisconnect);
            resolve();
        };
        events
            .on('subscribe', onSubscribe)
            .on('unsubscribe', onUnsubscribe)
            .on('publish', onPublish)
            .on('disconnect', onDisconnect)
            .onCleanup(onDone);
        connection
            .on('incoming-packet', async (packet) => {
            if (packet.cmd === 'connack') {
                const { properties, ...remainingPacket } = packet;
                const processedPacket = remainingPacket;
                if (properties) {
                    const { authenticationData, ...remainingProperties } = properties;
                    processedPacket.properties = remainingProperties;
                    if (authenticationData) {
                        processedPacket.properties.authenticationData =
                            authenticationData.toString('utf8');
                    }
                }
                events.emit('connected', {
                    url: item.payload.url,
                    packet: processedPacket,
                    connected: true,
                });
                item.payload.topics?.forEach((topic) => {
                    if (!topic.subscribe) {
                        return;
                    }
                    connection.subscribe(topic.name, { qos: topic.qos ?? 0 });
                });
                return;
            }
            if (packet.cmd === 'publish') {
                // Check if the packet has a content type.
                let contentType;
                try {
                    contentType = await (0, runtime_mqtt_utils_1.getContentType)(packet);
                }
                catch (err) {
                    events.emit('error', {
                        error: `Unable to detect content type: ${err}`,
                    });
                    return;
                }
                const { payload, properties, ...remainingPacket } = packet;
                const newPacket = {
                    ...remainingPacket,
                    // @ts-ignore - this should be sent as a buffer but json schema doesn't support it?
                    payload,
                };
                if (packet.properties && newPacket) {
                    const { correlationData, ...properties } = packet.properties;
                    newPacket.properties = properties;
                    if (correlationData) {
                        newPacket.properties.correlationData =
                            correlationData.toString('utf8');
                    }
                }
                events.emit('incoming-packet', {
                    packet: newPacket,
                    contentType,
                });
                return;
            }
            if (packet.cmd === 'suback') {
                // Check if the packet has a subscription map entry.
                if (!packet.messageId || !subscriptionMap.has(packet.messageId)) {
                    return;
                }
                const subscriptions = subscriptionMap.get(packet.messageId);
                // Update the packet with the topic names
                if (!subscriptions)
                    return;
                const grantedTopic = subscriptions.map((subscription, index) => {
                    return { topic: subscription.topic, qos: packet.granted[index] };
                });
                events.emit('incoming-packet', {
                    packet: { ...packet, granted: grantedTopic },
                });
                subscriptionMap.delete(packet.messageId);
                return;
            }
            if (packet.cmd === 'unsuback') {
                if (!packet.messageId || !unsubscriptionMap.has(packet.messageId)) {
                    return;
                }
                const unsubscriptions = unsubscriptionMap.get(packet.messageId);
                if (!unsubscriptions)
                    return;
                let ungrantedTopic = [];
                unsubscriptions.forEach((topic) => {
                    // MQTT.js doesn't seem to return a reason code for unsuback packets that can be used to check for errors.
                    ungrantedTopic = [{ topic }];
                });
                events.emit('incoming-packet', {
                    packet: { ...packet, granted: ungrantedTopic },
                });
                unsubscriptionMap.delete(packet.messageId);
                return;
            }
            if (packet.cmd === 'puback' ||
                packet.cmd === 'pubrel' ||
                packet.cmd === 'pubrec' ||
                packet.cmd === 'pubcomp') {
                events.emit('incoming-packet', { packet });
                return;
            }
            if (packet.cmd === 'disconnect') {
                if (packet.reasonCode && packet.reasonCode <= 25) {
                    events.emit('disconnected', {
                        packet,
                        message: packet.reasonCode === 0 || packet.reasonCode === 4 ?
                            'Disconnected from broker.'
                            : undefined,
                    });
                    receivedDisconnect = true;
                }
                // There was an error on disconnect. This is handled by the error handler
                return;
            }
            if (packet.cmd === 'pingreq' ||
                packet.cmd === 'pingresp' ||
                packet.cmd === 'auth') {
                // Adding this non function block to indicate explicitly that these packets are not handled.
                return;
            }
        })
            .on('outgoing-packet', (packet) => {
            if (packet.cmd === 'publish') {
                events.emit('outgoing-packet', {
                    packet: {
                        ...packet,
                        payload: packet.payload.toString('utf8'),
                    },
                });
                return;
            }
            if (!packet.messageId)
                return;
            if (packet.cmd === 'subscribe') {
                subscriptionMap.set(packet.messageId, packet.subscriptions);
                events.emit('outgoing-packet', { packet });
                return;
            }
            if (packet.cmd === 'unsubscribe') {
                unsubscriptionMap.set(packet.messageId, packet.unsubscriptions);
                return;
            }
        })
            .on('mqtt-error', (err) => {
            events.emit('error', { error: `An error occurred: ${err.message}` });
        })
            .on('error', (err) => {
            events.emit('error', { error: `An error occurred: ${err.message}` });
            connection.disconnect();
        })
            .on('reconnect', () => {
            events.emit('reconnecting', { message: 'Reconnecting to the broker.' });
        })
            .on('close', () => {
            if (!receivedDisconnect) {
                events.emit('disconnected', {
                    message: 'Disconnected from broker.',
                });
            }
            onDone();
        });
    });
});
//# sourceMappingURL=handler.js.map