"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.transformUserProperties = exports.validateSettings = void 0;
// Check MQTT client options and validate them
function validateSettings(url, mqttOptions) {
    if (!mqttOptions) {
        return {
            protocol: url.protocol,
            clean: false,
        };
    }
    // Check if there is a conflict between the clean setting and the clientId
    if (!mqttOptions.clientId && mqttOptions.cleanSession) {
        throw new Error('Unable to set clean to true when clientId is not provided');
    }
    // Extract options and properties that need to be updated for use by the client
    const { cleanSession, properties, lastWill, version, ...restOptions } = mqttOptions;
    const { userProperties, ...restProperties } = properties || {};
    const { payload, properties: willProperties, ...will } = lastWill || {};
    const { userProperties: willUserProperties, correlationData, ...restWillProperties } = willProperties || {};
    const options = restOptions;
    options.protocol = url.protocol;
    options.clean = mqttOptions.cleanSession ?? false;
    options.protocolVersion = version ?? 4;
    if (lastWill) {
        if (!lastWill.topic) {
            throw new Error('Will topic is required');
        }
        options.will = {
            payload: Buffer.from(payload ?? ''),
            ...will,
            topic: lastWill.topic,
        };
    }
    // Update properties and will properties if the protocol version is 5
    if (version === 5) {
        const transformedUserProperties = transformUserProperties(userProperties);
        options.properties = {
            ...restProperties,
            // Spreading empty object to avoid undefined properties
            ...(Object.keys(transformedUserProperties).length !== 0 ?
                { userProperties: transformedUserProperties }
                : {}),
        };
        if (options.will) {
            const transformedWillUserProperties = transformUserProperties(willUserProperties);
            options.will = {
                ...options.will,
                properties: {
                    ...restWillProperties,
                    ...(correlationData ?
                        { correlationData: Buffer.from(correlationData) }
                        : {}),
                    ...(Object.keys(transformedWillUserProperties).length != 0 ?
                        { userProperties: transformedWillUserProperties }
                        : {}),
                },
            };
        }
    }
    return optionsTypeGuard(options) ?
        { ...options }
        : { protocol: url.protocol ?? 'mqtt', clean: false };
}
exports.validateSettings = validateSettings;
// Transform UserProperties into an object with key-value pairs
function transformUserProperties(userProperties) {
    if (!userProperties || userProperties.length === 0)
        return {};
    return userProperties.reduce((acc, { key, value, disabled = false }) => {
        if (!disabled && value && value !== '') {
            acc[key] = value.toString();
        }
        return acc;
    }, {});
}
exports.transformUserProperties = transformUserProperties;
// Type guard for ConnectionOptions
function optionsTypeGuard(options) {
    return (options instanceof Object &&
        'protocol' in options &&
        Boolean(options?.protocol) &&
        'clean' in options);
}
//# sourceMappingURL=utilities.js.map