"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getSandbox = void 0;
const postman_sandbox_1 = __importDefault(require("postman-sandbox"));
const runtime_core_1 = require("@postman/runtime.core");
const async_task_queue_1 = __importDefault(require("./async-task-queue"));
const auto_increment_id_1 = __importDefault(require("./auto-increment-id"));
const sandboxMap = new WeakMap();
async function getSandbox(runContext) {
    let sandbox = sandboxMap.get(runContext);
    if (!sandbox) {
        sandbox = await new SandboxClient(runContext).init();
        sandboxMap.set(runContext, sandbox);
    }
    return sandbox;
}
exports.getSandbox = getSandbox;
class SandboxClient {
    constructor(runContext) {
        this.runContext = runContext;
        this.asyncScriptError = null;
        this.source = null;
        this.queue = new async_task_queue_1.default();
        this.cleanups = [];
        this.idGenerator = new auto_increment_id_1.default();
    }
    async init() {
        this.sandboxFleet = await promisify(postman_sandbox_1.default.createContextFleet)({}, { disabledAPIs: ['cookies', 'visualizer', 'execution'] }, { serializeLogs: true });
        this.runContext.events.onCleanup(() => this.dispose());
        return this;
    }
    register(templateName, template) {
        if (this.sandboxFleet.isRegistered(templateName)) {
            return;
        }
        this.sandboxFleet.register(templateName, template);
    }
    async execute(templateName, event, itemContext) {
        // Add validation on arguments
        // TODO: Cast event into PostmanEvent
        if (!event.script?.exec?.trim()) {
            return;
        }
        const task = () => this._execute(templateName, event, {
            ...itemContext,
            ...variablesToSandbox(this.runContext),
        });
        return new Promise((resolve) => {
            this.queue.push(task, () => {
                this.source = null;
                resolve();
            });
        });
    }
    async waitForCompletion() {
        await this.queue.drain();
    }
    get isStopped() {
        return this.runContext.events.closed;
    }
    async getContext(templateName) {
        return promisify(this.sandboxFleet.getContext.bind(this.sandboxFleet))(templateName);
    }
    emit(type, data, stopRun = false) {
        if (this.isStopped)
            return;
        if (this.source === null) {
            throw new Error('Cannot emit event without a source');
        }
        const event = { data, source: this.source };
        if (stopRun)
            event[runtime_core_1.STOP_RUN] = stopRun;
        this.runContext.events.emit(type, event);
    }
    attachListeners(context, eventId) {
        // ASSERTION
        const assertion = `execution.assertion.${eventId}`;
        context.on(assertion, (_, assertions) => {
            this.emit('extension:events:assertion', { assertions });
        });
        this.cleanups.push(() => context.removeAllListeners(assertion));
        // ERROR
        const asyncError = `execution.error.${eventId}`;
        context.once(asyncError, (_, e) => {
            // Caching the error originating from execution of async scripts.
            // These errors do not show up in the callback of `sandbox.execute()`.
            // Using `once` so that we only store the first error in case of multiple errors.
            this.asyncScriptError = e;
        });
        this.cleanups.push(() => context.removeAllListeners(asyncError));
        // CONSOLE
        // NOTE: This is global listener and might cause issues
        // if we enable parallel execution of scripts on same context.
        context.on('console', (_, level, messages) => {
            this.emit('extension:events:console', { level, messages });
        });
        this.cleanups.push(() => context.removeAllListeners('console'));
        // HTTP REQUEST
        const httpRequest = `execution.request.${eventId}`;
        context.on(httpRequest, (_, id, eventId, request) => {
            this.emit('extension:events:http-request', { id, eventId, request });
        });
        this.cleanups.push(() => context.removeAllListeners(httpRequest));
        // HTTP RESPONSE
        const { events } = this.runContext;
        const httpResponse = 'extension:events:http-response';
        const sandboxHttpResponse = `execution.response.${eventId}`;
        const onHTTPResponse = ({ payload }) => {
            if (this.isStopped)
                return;
            if (payload.id !== eventId)
                return;
            context.dispatch(sandboxHttpResponse, payload.eventId, payload.err, payload.response);
        };
        events.on(httpResponse, onHTTPResponse);
        this.cleanups.push(() => events.off(httpResponse, onHTTPResponse));
        this.cleanups.push(() => context.removeAllListeners(sandboxHttpResponse));
    }
    removeListeners() {
        this.cleanups.forEach((cleanup) => cleanup());
    }
    async _execute(templateName, event, executionContext) {
        if (this.isStopped)
            return;
        let err = null;
        let result = null;
        const eventId = this.idGenerator.generateID();
        this.source = {
            id: eventId,
            name: event.listen,
        };
        this.emit('extension:events:exec-begin');
        const sandboxContext = await this.getContext(templateName);
        if (this.isStopped)
            return;
        this.attachListeners(sandboxContext, eventId);
        try {
            result = await promisify(sandboxContext.execute.bind(sandboxContext))(event, {
                id: eventId,
                context: executionContext,
                legacy: {
                    _itemId: executionContext?.request?.id,
                    _itemName: executionContext?.request?.name,
                    // TODO: populate these values
                    _itemPath: null,
                    _eventItemName: null,
                },
            });
        }
        catch (e) {
            err = e;
        }
        this.removeListeners();
        if (this.isStopped)
            return;
        if (err || this.asyncScriptError) {
            this.emit('extension:events:error', { error: err || this.asyncScriptError }, true);
            this.asyncScriptError = null;
            return;
        }
        if (result) {
            const variables = variablesFromSandbox(result);
            objectKeys(variables).forEach((key) => this.runContext[key].put(variables[key]));
        }
        this.emit('extension:events:exec-end', { result });
    }
    dispose() {
        this.queue.kill();
        this.removeListeners();
        this.sandboxFleet.disposeAll();
        sandboxMap.delete(this.runContext);
    }
}
function promisify(fn) {
    return (...args) => new Promise((resolve, reject) => {
        fn(...args, (err, result) => {
            if (err) {
                return reject(err);
            }
            resolve(result);
        });
    });
}
const variablesToSandbox = (runContext) => {
    return {
        _variables: runContext.locals.toJSON(),
        environment: runContext.environment.toJSON(),
        globals: runContext.globals.toJSON(),
        collectionVariables: runContext.collectionVariables.toJSON(),
    };
};
const variablesFromSandbox = (result) => {
    return {
        locals: result._variables?.values,
        environment: result.environment?.values,
        globals: result.globals?.values,
        collectionVariables: result.collectionVariables?.values,
    };
};
// Like Object.keys, but unsound in interest of more convenience
const objectKeys = (obj) => Object.keys(obj);
//# sourceMappingURL=index.js.map